﻿package jp.progression.commands.tweens {
	import com.greensock.TweenMax;
	import jp.nium.core.debug.Logger;
	import jp.nium.utils.ObjectUtil;
	import jp.progression.commands.Command;
	import jp.progression.commands.CommandInterruptType;
	import jp.progression.core.L10N.L10NCommandMsg;
	import jp.progression.events.ExecuteEvent;
	
	/**
	 * TweenMax.fromToのコマンド
	 * 
	 * @author ...
	 */
	public class DoTweenMaxFromTo extends Command {
		
		private var _target:Object;	//トゥイーンの対象
		private var _duration:Number;	//効果時間
		private var _fromParameters:Object;	//パラメーター。from
		private var _toParameters:Object;	//パラメーター。to
		private var _parametersTween:Object;	//パラメーター。トゥイーン用?
		private var _originalParameters:Object;	//実行前のパラメーターを保持
		private var _onUpdate:Function;
		
		
		//--------------------------------------------------------------------------
		//
		//  メソッド
		//
		//--------------------------------------------------------------------------
		
		/**
		 * com.greensok.TweenMax パッケージのイージング機能を実行するコマンドクラスです。
		 * 
		 * @param target Target object whose properties this tween affects. This can be ANY object, not just a DisplayObject. 
		 * @param duration Duration in seconds (or in frames for frames-based tweens)
		 * @param fromVars An object containing the starting values of the properties you're tweening. For example, to tween from x=0, y=0, you could pass {x:0, y:0}. Only put starting values in the fromVars parameter - all special properties for the tween (like onComplete, onUpdate, delay, etc.) belong in the toVars parameter. 
		 * @param toVars An object containing the ending values of the properties you're tweening. For example, to tween to x=100, y=100, you could pass {x:100, y:100}. It can also contain special properties like "onComplete", "ease", "delay", etc.
		 * @param	initObject - 設定したいプロパティを含んだオブジェクトです。
		 */
		public function DoTweenMaxFromTo(target:Object, duration:Number, fromVars:Object, toVars:Object, initObject:Object = null) {
			// 引数を設定する
			_target = target;
			_duration = duration;
			_fromParameters = fromVars || { };
			_toParameters = toVars || { };
			
			// 親クラスを初期化します。
			super(_executeFunction, _interruptFunction, initObject);
			
			// initObject が DoTweenMaxTo であれば
			var com:DoTweenMaxFromTo = initObject as DoTweenMaxFromTo;
			if (com)	_onUpdate = com._onUpdate;	// 特定のプロパティを継承する
		}
		
		/**
		 * 実行されるコマンドの実装です。
		 */
		private function _executeFunction():void {
			// 既存のイベントハンドラメソッドが存在したら例外をスローする
			switch ( true ) {
				case _toParameters.onInit:
				case _toParameters.onInitParams:
				case _toParameters.onStart:
				case _toParameters.onStartParams:
				case _toParameters.onUpdae:
				case _toParameters.onUpdateParams:
				case _toParameters.onComplete:
				case _toParameters.onCompleteParams:
				case _toParameters.onReverseComplete:
				case _toParameters.onReverseCompleteParams:
					throw new Error( Logger.getLog( L10NCommandMsg.getInstance().ERROR_001 ).toString() );
					break;
			}
			
			// 現在の状態を保存する
			_originalParameters = [];
			_parametersTween = {};
			for (var p:String in _toParameters) {
				_parametersTween[p] = _toParameters[p];
				
				if (p in _target)	_originalParameters[p] = _target[p];
			}
			
			// 初期化する
			_parametersTween.onComplete = _complete;
			_parametersTween.onUpdate = _update;
			
			// 実行する
			TweenMax.fromTo(_target, _duration, _fromParameters, _parametersTween);
		}
		
		/**
		 * トゥイーン中
		 * 
		 */
		private function _update():void {
			// イベントを送出する
			super.dispatchEvent(new ExecuteEvent(ExecuteEvent.EXECUTE_UPDATE, false, false, this));
			
			// イベントハンドラメソッドを実行する
			if (_onUpdate != null)	_onUpdate.apply(super.scope || this);
		}
		
		/**
		 * 終了
		 * 
		 */
		private function _complete():void {
			
			// 破棄する
			_destroy();
			
			// 処理を終了する
			if ( super.state > 0 ) {
				super.executeComplete();
			}
			else	_interruptFunction();
		}
		
		/**
		 * 破棄します。
		 */
		private function _destroy():void {
			// 破棄する
			_target = null;
			_parametersTween = null;
			_originalParameters = null;
		}
		
		/**
		 * 中断されるコマンドの実装です。
		 */
		private function _interruptFunction():void {
			
			// 中断する
			try {
				TweenMax.killTweensOf(_target);
			}
			catch ( e:Error ) {}
			
			// 中断方法によって処理を振り分ける
			switch (super.interruptType) {
				case CommandInterruptType.ABORT: {
					//処理が実行される以前の状態に戻すように指定します。
					TweenMax.to(_target, 0, _fromParameters);
					break;
				}
				case CommandInterruptType.SKIP: {
					//処理が完了された状態と同様になるように指定します。
					TweenMax.to(_target, 0, _toParameters);
					break;
				}
			}
		}
		
		/**
		 * インスタンスのコピーを作成して、各プロパティの値を元のプロパティの値と一致するように設定します。
		 */
		override public function clone():Command {
			return new DoTweenMaxFromTo(_target, _duration, _fromParameters, _toParameters, this);
		}
	}
}
