﻿package jp.progression.commands.tweens {
	import com.greensock.TweenMax;
	import jp.nium.core.debug.Logger;
	import jp.nium.utils.ObjectUtil;
	import jp.progression.commands.Command;
	import jp.progression.commands.CommandInterruptType;
	import jp.progression.core.L10N.L10NCommandMsg;
	import jp.progression.events.ExecuteEvent;
	
	/**
	 * TweenMax.fromのコマンド
	 * 
	 * @author ...
	 */
	public class DoTweenMaxFrom extends Command {
		
		private var _target:Object;	//トゥイーンの対象
		private var _duration:Number;	//効果時間
		private var _parameters:Object;	//パラメーター
		private var _parametersTween:Object;	//パラメーター。トゥイーン用?
		private var _originalParameters:Object;	//実行前のパラメーターを保持
		private var _onUpdate:Function;
		
		
		//--------------------------------------------------------------------------
		//
		//  メソッド
		//
		//--------------------------------------------------------------------------
		
		/**
		 * com.greensok.TweenMax パッケージのイージング機能を実行するコマンドクラスです。
		 * 
		 * @param	target - Target object whose properties this tween affects. This can be ANY object, not just a DisplayObject. 
		 * @param	duration - Duration in seconds (or in frames for frames-based tweens)
		 * @param	vars - An object containing the start values of the properties you're tweening. For example, to tween from x=100, y=100, you could pass {x:100, y:100}. It can also contain special properties like "onComplete", "ease", "delay", etc.
		 * @param	initObject - 設定したいプロパティを含んだオブジェクトです。
		 */
		public function DoTweenMaxFrom(target:Object, duration:Number, vars:Object, initObject:Object = null) {
			// 引数を設定する
			_target = target;
			_duration = duration;
			_parameters = vars || { };
			
			// 親クラスを初期化します。
			super(_executeFunction, _interruptFunction, initObject);
			
			// initObject が DoTweenMaxFrom であれば
			var com:DoTweenMaxFrom = initObject as DoTweenMaxFrom;
			if (com)	_onUpdate = com._onUpdate;	// 特定のプロパティを継承する
		}
		
		/**
		 * 実行されるコマンドの実装です。
		 */
		private function _executeFunction():void {
			// 既存のイベントハンドラメソッドが存在したら例外をスローする
			switch ( true ) {
				case _parameters.onInit:
				case _parameters.onInitParams:
				case _parameters.onStart:
				case _parameters.onStartParams:
				case _parameters.onUpdae:
				case _parameters.onUpdateParams:
				case _parameters.onComplete:
				case _parameters.onCompleteParams:
				case _parameters.onReverseComplete:
				case _parameters.onReverseCompleteParams:
					throw new Error( Logger.getLog( L10NCommandMsg.getInstance().ERROR_001 ).toString() );
					break;
			}
			
			// 現在の状態を保存する
			_originalParameters = [];
			_parametersTween = {};
			for (var p:String in _parameters) {
				_parametersTween[p] = _parameters[p];
				
				if (p in _target)	_originalParameters[p] = _target[p];
			}
			
			// 初期化する
			_parametersTween.onComplete = _complete;
			_parametersTween.onUpdate = _update;
			
			// 実行する
			TweenMax.from(_target, _duration, _parametersTween);
		}
		
		/**
		 * トゥイーン中
		 * 
		 */
		private function _update():void {
			// イベントを送出する
			super.dispatchEvent(new ExecuteEvent(ExecuteEvent.EXECUTE_UPDATE, false, false, this));
			
			// イベントハンドラメソッドを実行する
			if (_onUpdate != null)	_onUpdate.apply(super.scope || this);
		}
		
		/**
		 * 終了
		 * 
		 */
		private function _complete():void {
			
			// 破棄する
			_destroy();
			
			// 処理を終了する
			if ( super.state > 0 ) {
				super.executeComplete();
			}
			else	_interruptFunction();
		}
		
		/**
		 * 破棄します。
		 */
		private function _destroy():void {
			// 破棄する
			_target = null;
			_parametersTween = null;
			_originalParameters = null;
		}
		
		/**
		 * 中断されるコマンドの実装です。
		 */
		private function _interruptFunction():void {
			
			// 中断する
			try {
				TweenMax.killTweensOf(_target);
			}
			catch ( e:Error ) {}
			
			// 中断方法によって処理を振り分ける
			switch (super.interruptType) {
				case CommandInterruptType.ABORT: {
					//処理が実行される以前の状態に戻すように指定します。
					TweenMax.to(_target, 0, _parameters);
					break;
				}
				case CommandInterruptType.SKIP: {
					//処理が完了された状態と同様になるように指定します。
					TweenMax.to(_target, 0, _originalParameters);
					break;
				}
			}
		}
		
		/**
		 * インスタンスのコピーを作成して、各プロパティの値を元のプロパティの値と一致するように設定します。
		 */
		override public function clone():Command {
			return new DoTweenMaxFrom(_target, _duration, _parameters, this);
		}
	}
}
